package gov.va.vamf.scheduling.batch;

import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.veteranappointment.datalayer.patient.PatientPreferenceDataLayerRepo;
import com.agilex.healthcare.veteranappointment.domain.VARPatientPreference;
import gov.va.vamf.scheduling.communitycare.service.CCService;
import gov.va.vamf.scheduling.communitycare.domain.BookedCCAppointment;

import gov.va.vamf.scheduling.batch.datalayer.reminder.RecallReminderDataService;
import gov.va.vamf.scheduling.batch.domain.RecallReminder;
import gov.va.vamf.scheduling.batch.domain.RecallReminderMessage;
import gov.va.vamf.scheduling.batch.domain.CCAppointmentReminder;
import gov.va.vamf.scheduling.batch.domain.Secret;
import gov.va.vamf.scheduling.batch.domain.Token;
import gov.va.vamf.scheduling.direct.datalayer.appointment.PatientIdentifiersDataService;
import org.apache.commons.lang.time.DateFormatUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Entity;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriBuilder;
import java.net.URI;
import java.util.*;

@Component
public class RecallReminderWorker {

    @Resource
    private RecallReminderDataService recallReminderDataService;

    @Resource
    PatientPreferenceDataLayerRepo patientPreferenceDataLayerRepo;

    @Resource
    private CCService ccService;

    @Resource
    PatientIdentifiersDataService patientIdentifiersDataService;

    @Value("${vmm.token.uri}")
    private String vmmTokenUri;

    @Value("${vmm.post.recall.reminder.uri}")
    private String vmmPostRecallReminderUri;

    @Value("${recall.reminder.email.subject}")
    private String recallReminderEmailSubject;

    @Value("${recall.reminder.communitycare.email.subject}")
    private String recallReimnderCommunityCareEmailSubject;

    @Value("${recall.reminder.email.body}")
    private String recallReminderEmailBody;

    @Value("${recall.reminder.communitycare.email.body}")
    private String recallReminderCommunityCareEmailBody;

    @Value("${recall.num.days}")
    private int recallNumDays;

    @Value("${var.secret}")
    private String varSecret;

    private final static Log logger = LogFactory.getLog(RecallReminderWorker.class);

    private static final String FACILITY_NAME = "<Facility Name>";
    private static final String CLINIC_NAME = "<Clinic Name>";
    private static final String CLINIC_FRIENDLY_NAME = "<Clinic location friendly name>";
    private static final String PROVIDER_NAME = "<Provider Name>";
    private static final String RECALL_DATE = "<recall date>";

    private static final String APPOINTMENT_DATE = "<Appointment Date>";
    private static final String APPOINTMENT_TIME = "<Appointment Time>";
    private static final String APPOINTMENT_TIME_ZONE = "<Appointment Time Zone>";

    private static final String PRACTICE_NAME = "<Practice Name>";
    private static final String PROVIDER_PHONE = "<Provider phone>";

    private static final String ICN    = "ICN";
    private static final String EDIPI  = "EDIPI";
    private static final String EMAIL  = "email";
    private static final String IN_APP = "in-app";

    private static final List<String> MSG_TYPE_BOTH = Collections.unmodifiableList(Arrays.asList(EMAIL, IN_APP));;
    private static final List<String> MSG_TYPE_IN_APP = Collections.singletonList(IN_APP);
    private static final List<String> MSG_TYPE_EMAIL = Collections.singletonList(EMAIL);

    public void processRecallReminders() {
        logger.info("Begin Recall Reminder processing " + new Date());

        Client client = getRestClient();

        try {
            List<RecallReminder> recallReminders = fetchRecallReminders(getRecallDate());

            String token = getToken(client);

            for(RecallReminder recallReminder : recallReminders) {
                postRecallReminderMessageToVmm(client, token, recallReminder);
            }
        } catch (Exception e) {
            logger.error(e.getMessage());
        } finally {
            client.close();
        }

        logger.info("End Recall Reminder processing " + new Date());
    }
    @Scheduled(cron = "${batch.recall.communityCare.cron.expression}")
    public void processCommunityCareAppointmentReminders(){
        logger.info("Begin Community Care Recall Reminder processing " + new Date());
        Client client = getRestClient();
        try {
            List<CCAppointmentReminder> ccAppointmentReminders = getAppointmentRequestMessages();

            String token = getToken(client);

            for(CCAppointmentReminder ccAppointmentReminder : ccAppointmentReminders) {
                postCommunityCareAppointmentReminderMessageToVmm(client, token, ccAppointmentReminder);
            }
        } catch (Exception e) {
            logger.error("Error processing community care appointment reminders", e);
        } finally {
            client.close();
        }

        logger.info("End Community Care Recall Reminder processing " + new Date());
    }

    public String getToken(Client client) {

        Secret secret = new Secret();
        secret.setSecret(varSecret);

        Response response = client.target(vmmTokenUri)
            .request()
            .accept(MediaType.APPLICATION_JSON)
            .put(Entity.json(secret));

        return response.readEntity(Token.class).getToken();
    }

    public void postRecallReminderMessageToVmm(Client client, String token, RecallReminder recallReminder) {

        defaultEmptyStrings(recallReminder);
        String titleSlashSubject = constructEmailSubject(recallReminder);

        RecallReminderMessage recallReminderMessage = new RecallReminderMessage();

        recallReminderMessage.setSource("VA Appointments - Follow-up Appointment Reminder");
        recallReminderMessage.setTypes(determineMessageType(recallReminder));
        recallReminderMessage.setTo(recallReminder.getEmailAddress());
        recallReminderMessage.setTitle(titleSlashSubject);
        recallReminderMessage.setSubject(titleSlashSubject);
        recallReminderMessage.setText(constructEmailMessageText(recallReminder));
        recallReminderMessage.setPatientIdentifier(new PatientIdentifier(ICN, recallReminder.getPatientIcn()));

        logger.info("Recall reminder for a patient being sent");

        client.target(vmmPostRecallReminderUri)
            .request()
            .accept(MediaType.APPLICATION_JSON)
            .header("Authorization", token)
            .post(Entity.json(recallReminderMessage));
    }

    public void postCommunityCareAppointmentReminderMessageToVmm(Client client, String token, CCAppointmentReminder ccAppointmentReminder) {

        RecallReminderMessage recallReminderMessage = new RecallReminderMessage();
        recallReminderMessage.setTypes(MSG_TYPE_EMAIL);
        recallReminderMessage.setSource(recallReimnderCommunityCareEmailSubject);
        recallReminderMessage.setTo(ccAppointmentReminder.getPatientEmailAddress());
        recallReminderMessage.setTitle(recallReimnderCommunityCareEmailSubject);
        recallReminderMessage.setSubject(recallReimnderCommunityCareEmailSubject);
        recallReminderMessage.setText(constructCommunityCareEmailMessageText(ccAppointmentReminder));
        recallReminderMessage.setPatientIdentifier(ccAppointmentReminder.getPatientIdentifier());

        logger.info("Community Care Recall reminder for a patient being sent");

        client.target(vmmPostRecallReminderUri)
                .request()
                .accept(MediaType.APPLICATION_JSON)
                .header("Authorization", token)
                .post(Entity.json(recallReminderMessage));
    }

    private List<String> determineMessageType(RecallReminder recallReminder) {

        PatientIdentifier icn = constructPatientIcnFromRecallReminder(recallReminder);
        PatientIdentifier edipi = patientIdentifiersDataService.getCorrelatingPatientIdentifier(icn, EDIPI);

        if (edipi != null) {
            VARPatientPreference varPatientPreference = patientPreferenceDataLayerRepo.fetchPatientPreference(edipi);

            if (varPatientPreference != null && varPatientPreference.isEmailAllowed()) {
                recallReminder.setEmailAddress(varPatientPreference.getEmailAddress());
                return MSG_TYPE_BOTH;
            }
        }
        return MSG_TYPE_IN_APP;
    }

    private PatientIdentifier constructPatientIcnFromRecallReminder(RecallReminder r) {
        return new PatientIdentifier("ICN", r.getPatientIcn() +
            (r.getPatientIcnChecksum().startsWith("V") ? r.getPatientIcnChecksum() : "V" + r.getPatientIcnChecksum()));
    }

    private void defaultEmptyStrings(RecallReminder recallReminder) {

        if (recallReminder.getRecallDate() == null) recallReminder.setRecallDate(new Date());
        if (recallReminder.getFacilityName() == null) recallReminder.setFacilityName("");
        if (recallReminder.getClinicName() == null) recallReminder.setClinicName("");
        if (recallReminder.getClinicFriendlyName() == null) recallReminder.setClinicFriendlyName("");
        if (recallReminder.getStaffName() == null) recallReminder.setStaffName("");
        if (recallReminder.getEmailAddress() == null) recallReminder.setEmailAddress("");
    }

    private String constructEmailSubject(RecallReminder recallReminder) {
        return recallReminderEmailSubject.replace(CLINIC_NAME, recallReminder.getClinicName());
    }

    private String constructEmailMessageText(RecallReminder recallReminder) {
        return recallReminderEmailBody.replace(RECALL_DATE, DateFormatUtils.format(recallReminder.getRecallDate(), "MM/dd/yyyy"))
            .replace(FACILITY_NAME, recallReminder.getFacilityName())
            .replace(CLINIC_NAME, recallReminder.getClinicName())
            .replace(CLINIC_FRIENDLY_NAME, recallReminder.getClinicFriendlyName())
            .replace(PROVIDER_NAME, recallReminder.getStaffName());
    }

    private String constructCommunityCareEmailMessageText(CCAppointmentReminder ccAppointmentReminder) {
        return recallReminderCommunityCareEmailBody
                .replace(APPOINTMENT_DATE, DateFormatUtils.format(ccAppointmentReminder.getAdjustedDateTime(), "MM/dd/yyyy"))
                .replace(APPOINTMENT_TIME, DateFormatUtils.format(ccAppointmentReminder.getAdjustedDateTime(), "HH:mm"))
                .replace(APPOINTMENT_TIME_ZONE, ccAppointmentReminder.getTimeZoneAbbr())
                .replace(PRACTICE_NAME, ccAppointmentReminder.getPracticeName())
                .replace(PROVIDER_PHONE, ccAppointmentReminder.getProviderPhoneNumber());
    }

    private Client getRestClient() {
        return ClientBuilder.newBuilder().build();
    }

    private URI buildUri(String url, Map<String, String> pathParams) {
        return UriBuilder.fromUri(url).buildFromMap(pathParams);
    }

    private Date getRecallDate() {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DAY_OF_YEAR, recallNumDays);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTime();
    }

    public List<RecallReminder> fetchRecallReminders(Date recallDate) {
        return recallReminderDataService.fetchRecallReminders(recallDate);
    }

    public List<RecallReminder> filterByPreference(List<RecallReminder> recallReminders) {
        List<RecallReminder> filteredRecallReminders = new ArrayList<RecallReminder>();

        for(RecallReminder r : recallReminders) {
            PatientIdentifier icn = constructPatientIcnFromRecallReminder(r);
            PatientIdentifier edipi = patientIdentifiersDataService.getCorrelatingPatientIdentifier(icn, "EDIPI");

            if (edipi != null) {
                VARPatientPreference varPatientPreference = patientPreferenceDataLayerRepo.fetchPatientPreference(edipi);

                if (varPatientPreference.isEmailAllowed()) {
                    r.setEmailAddress(varPatientPreference.getEmailAddress());
                    filteredRecallReminders.add(r);
                }

                if (varPatientPreference.isTextMsgAllowed()) {
                    r.setTextMessagingPhoneNumber(varPatientPreference.getTextMsgPhNumber());
                    filteredRecallReminders.add(r);
                }
            }
        }

        return filteredRecallReminders;
    }

    public String getVmmTokenUri() {
        return vmmTokenUri;
    }

    public String getVmmPostRecallReminderUri() {
        return vmmPostRecallReminderUri;
    }

    public String getRecallReminderEmailSubject() {
        return recallReminderEmailSubject;
    }

    public String getRecallReminderEmailBody() {
        return recallReminderEmailBody;
    }

    public int getRecallNumDays() {
        return recallNumDays;
    }

    /**
     * A resource that will fetch a list of upcoming communitycare appointments to send out
     */
    private ArrayList<CCAppointmentReminder> getAppointmentRequestMessages() {

        List<BookedCCAppointment> allAppointmentsList = ccService.findUpComingAppointments();
        ArrayList<CCAppointmentReminder> ccAppointmentReminderList = new ArrayList<>();
        // HashMap(int initialCapacity, float loadFactor)
        // No rehash operations will ever occur if the initial capacity is greater than the maximum number of entries the Hashtable will contain divided by its load factor.
	    HashMap<String, String> emailList = new HashMap<>(allAppointmentsList.size()+1, 1);
        for(BookedCCAppointment ccAppointment : allAppointmentsList) {
            //ICN is assigningAuthority
            String uniqueId = ccAppointment.getPatientIdentifier().getUniqueId();
	        CCAppointmentReminder ccAppointmentReminder = new CCAppointmentReminder(ccAppointment);

            if(isDateInNotificationRange(ccAppointmentReminder.getAdjustedDateTime())) {
                if (!emailList.containsKey(uniqueId)) {
                    VARPatientPreference varPatientPreference = ccService.fetchPatientPreference(uniqueId);
                    if (varPatientPreference.isEmailAllowed() && !varPatientPreference.getNotificationFrequency().equalsIgnoreCase("Never")) {
                        ccAppointmentReminder.addNotificationPreferenceDetails(varPatientPreference);
                        ccAppointmentReminderList.add(ccAppointmentReminder);
                        emailList.put(uniqueId, ccAppointmentReminder.getPatientEmailAddress());
                    } else {
                        emailList.put(uniqueId, null);
                    }
                } else if (emailList.get(uniqueId)!= null) {
                    ccAppointmentReminder.setPatientEmailAddress(emailList.get(uniqueId));
                    ccAppointmentReminderList.add(ccAppointmentReminder);
                }
            }
        }
        return ccAppointmentReminderList;
    }

    private boolean isDateInNotificationRange(Date appointmentDate){
        int[] dates = {0,1,3,7};
        Calendar appointment = new GregorianCalendar();
        appointment.setTime(appointmentDate);
        Calendar testDates = Calendar.getInstance();
        Date today = new Date();
        for(int additionalDays: dates) {
            testDates.setTime(today);
            testDates.add(Calendar.DAY_OF_YEAR, additionalDays);
            if (testDates.get(Calendar.YEAR) == appointment.get(Calendar.YEAR) && testDates.get(Calendar.DAY_OF_YEAR) == appointment.get(Calendar.DAY_OF_YEAR)) {
                return true;
            }
        }
        return false;
    }
}
